define(['bower_components/angular/angular', 'app'], function(angular, app) {
	"use strict";
	app.service('AbstractGoalsService', function(CollectionCRUDService, formatter, $filter, localResourceDirectoryService) {
		var AbstractGoalsService = function() {

			var config = {},
				fhirDataItem = null;

			config.dependency = {
				service: localResourceDirectoryService,
				linkTitles: {
					GET: "goal",
					POST: "goal"
				}
			};

			config.collection = {
				name: "goals",
				objectType: "Goals",
				sortComparator: function(left, right) {
					if (left === right) {
						return 0;
					}

					var leftDate = new Date(left.resource.startDate),
						rightDate = new Date(right.resource.endDate);

					if (leftDate < rightDate) {
						return 1;
					} else {
						return -1;
					}
				},
				filter: function(list, filters) {
					if (!filters || !list) {
						return list;
					}

					if (filters.status) {
						var fhirResponseStatus = resolveStatus(filters),
						FiltersStatus = filters.status,
						operationOutcomeFlag = false;

						return list.filter(function(item) {
							if (item.resourceType === "OperationOutcome") {
								operationOutcomeFlag = true;
							}
							if (operationOutcomeFlag) {
								return false;
							}
							return (item.resource.status.toUpperCase() === fhirResponseStatus.toUpperCase()) ? true : (item.resource.status.toUpperCase() === FiltersStatus.toUpperCase());
						});
					}

					return list;
				}
			};

			config.object = {
				uidKey: "resource.id",
				key: "goal",
				responseTransform: function(item) {
					item.status = resetStatus(item);
					item.startDate = formatter.getFormattedFrontendDate(item.startDate);
					if (item.targetDate) {
						item.endDate = formatter.getFormattedFrontendDate(item.targetDate);
					}
					if (item.statusDate) {
						item.statusDate = formatter.getFormattedFrontendDate(item.statusDate);
					}

					if (!item.note) {
						item.note = [{
							time: null,
							text: null,
							authorString: false
						}];
					}

					if (item.description === "<blank description value>") {
						delete item.description;
					}

					item.note.forEach(function(step) {
						if (step.time) {
							step.time = formatter.getFormattedFrontendDate(step.time);
						}
						if (step.authorString === "progress") {
							step.text = parseInt(step.text);
							item.progresses.push(step);
						} else if (step.authorString) {
							// remap string back to boolean
							step.authorString === "true" ? step.authorString = true : step.authorString = step.authorString;
							step.authorString === "false" ? step.authorString = false : step.authorString = step.authorString;
						}
					});

					for (var i = 0; i < item.note.length; i++) {
						var step = item.note[i];
						if (step.authorString === "progress") {
							item.note.splice(i, 1);
							i--;
						}
					}

					if (!item.completedDate) {
						delete item['completedDate'];
					} else {
						item.completedDate = formatter.getFormattedFrontendDate(item.completedDate);
					}
					return item;
				},
				requestTransform: function(item) {
					var currentDate = new Date();
					item.resourceType = "Goal";
					item.subject = {
						reference: "Patient/" + this.patient.id
					};
					if (_.get(item, "extension[0].valueString")) {
						item.extension[0].url = "http://pgdfhir.vamobile.us/fhir/StructureDefinition/goal-reward";
						item.extension = [{
							url: item.extension[0].url,
							valueString: item.extension[0].valueString
						}];
					} else {
						delete item.extension;
					}

					item.identifier = [{
						value: item.identifier[0].value
					}];
					item.startDate = formatter.getFormattedBackendDate(item.startDate);
					item.targetDate = formatter.getFormattedBackendDate(item.endDate);
					item.status = resolveStatus(item);
					getComponents(item);

					delete item['endDate'];

					for (var i = 0; i < item.note.length; i++) {
						var step = item.note[i];

						// authorString === step complete checkbox in UI or progress bars
						if (_.has(step, 'authorString')) {
							if (step.authorString === "progress") {
								step.text = parseInt(step.text);
								item.progresses.push(step);
							} else if (step.authorString) {
								// must convert boolean to string for backend validation to pass
								step.authorString = "true";
							} else {
								step.authorString = "false"
							}
						}

						if (step.text) {
							step.time = formatter.getFormattedBackendDateTime(step.time);
						} else {
							item.note.splice(i, 1);
							i--;
						}
					}

					if (item.progresses) {
						item.progresses.forEach(function(progress) {
							progress.time = formatter.getFormattedBackendDate(progress.time);
							item.note.push(progress)
						});
						delete item.progresses;
					} else {
						delete item.progresses;
					}

					if (item.note.length === 0) {
						delete item.note;
					}

					if (item.createdDate) {
						delete item.createdDate;
					}

					if (item.notificationStatus) {
						delete item.notificationStatus;
					}

					if (!item.description) {
						item.description = "<blank description value>";
					}

					if (item.statusDate) {
						item.statusDate = formatter.getFormattedBackendDate(item.statusDate);
					} else {
						item.statusDate = "";
					}

					return item;
				},
				queryParamsFhirTransform: function(queryParams) {
					var fhirParams = {};

					if (queryParams && queryParams.subject) {
						fhirParams.subject = "Patient/" + this.patient.id;
					}
					if (queryParams.startDate && queryParams.endDate) {
						fhirParams.targetdate = [">=" + formatter.getFormattedBackendDateTime(queryParams.startDate+" 00:00:00"),
							"<=" + formatter.getFormattedBackendDateTime(queryParams.endDate+" 23:59:59")];
					}
					if (queryParams.status) {
						fhirParams.status = resolveStatus(queryParams);
					}
					if (queryParams.type) {
						fhirParams.category = queryParams.type.toLowerCase();
					}
					fhirParams['_count'] = 100;

					return fhirParams;
				}
			};

			var getComponents = function(item) {
				if (typeof item.category === "object" && item.category.length > 0 && item.category[0].text) {
					item.category = [{text : item.category[0].text}];
					switch ('Health') {
						case 'Health':
							item.category[0].coding = [{
								system: "http://hl7.org/fhir/goal-category",
								 code: "health",
								display: "Health"
							}];
							break;
						case 'Finance':
							item.category[0].coding = [{
								system: "http://hl7.org/fhir/goal-category",
								 code: "finance",
								display: "Finance"
							}];
							break;
						case 'Work':
							item.category[0].coding = [{
								system: "http://hl7.org/fhir/goal-category",
								 code: "work",
								display: "Work"
							}];
							break;
						case 'Leisure':
							item.category[0].coding = [{
								system: "http://hl7.org/fhir/goal-category",
								 code: "leisure",
								display: "Leisure"
							}];
							break;
						case 'Relationships':
							item.category[0].coding = [{
								system: "http://hl7.org/fhir/goal-category",
								 code: "relationships",
								display: "Relationships"
							}];
							break;
						default:
							item.category[0].coding = [{
								system: "http://hl7.org/fhir/goal-category",
								 code: "other",
								display: "Other"
							}];

					}
				}else {
					delete item.category
				}

				return item;
			};

			var resetStatus = function(item) {
				var statusList = [{"in-progress": "Active"},{"on-hold": "Inactive"},{"achieved": "Complete"}];
				var tStatusItem = statusList.filter(function(listItem) { return listItem[item.status]; });
				return (tStatusItem && tStatusItem.length > 0) ? tStatusItem[0][item.status] : item.status;
			};

			var resolveStatus = function(item) {
				var statusList = [{Active: "in-progress"},{Inactive: "on-hold"},{Complete: "achieved"}];
				var tStatusItem = statusList.filter(function(listItem) { return listItem[item.status]; });
				return (tStatusItem && tStatusItem.length > 0) ? tStatusItem[0][item.status] : item.status;
			};

			CollectionCRUDService.call(this, config);

			this.createEmpty = function() {
				var empty = {
					status: "Active",
					statusDate: formatter.getFormattedFrontendDate(new Date()),
					category: [],
					progresses: [],
					startDate: formatter.getFormattedFrontendDate(new Date()),
					endDate: null,
					notificationStatus: "Off"
				};

				return this.config.object.responseTransform(empty);
			};

			this.refresh = function(content) {
				var that = this;
				that.config.object.responseTransform(content);
				if (content.id) {
					that._deleteFromLocalList({
						resource: content
					});
				}

				var scopedList,
					unfilteredList;

				if (that.dataPointers.list) {
					scopedList = angular.copy(that.dataPointers.list);

					scopedList.push({
						resource: content
					});

					scopedList = that.config.collection.filter ? that.config.collection.filter(scopedList, that.dataPointers.filters) : scopedList;
					scopedList = scopedList.sort(that.config.collection.sortComparator);
				}

				if (that.dataPointers.unfilteredList) {
					unfilteredList = angular.copy(that.dataPointers.unfilteredList);

					unfilteredList.push({
						resource: content
					});

					unfilteredList = unfilteredList.sort(that.config.collection.sortComparator);
				}

				that.dataPointers.list = scopedList;
				that.dataPointers.unfilteredList = unfilteredList;
			};

			this.deleteFromList = function(item) {
				var that = this;
				that._deleteFromLocalList({
					resource: item
				});
			};

			this.fetch = function(queryParams) {
				var that = this;
				var fhirParams = that.config.object.queryParamsFhirTransform(queryParams);
				var requestPromise = that._onRequestComplete(that.BaseCRUDService.fetch(fhirParams));

				requestPromise.then(function(response) {
					that.dataPointers.filters = angular.copy(queryParams);
					if (response.resourceType === "Bundle" && response.entry) {
						that.dataPointers.unfilteredList = response.entry.sort(that.config.collection.sortComparator);
						that.dataPointers.list = that.config.collection.filter(response.entry, queryParams).sort(that.config.collection.sortComparator);
					}
					else {
						that.dataPointers.list = [];
						that.dataPointers.unfilteredList = [];
					}
					that.dataPointers.link = response.link;
					response.size = that.dataPointers.list.length;
					for (var item in that.dataPointers.list) {
						var resource = that.dataPointers.list[item].resource;
						resource.progresses = [];
						that.config.object.responseTransform(resource);
					}
				});

				return requestPromise;
			};

			this.save = function(item) {
				var that = this;
				var requestPromise = that._onRequestComplete(that.BaseCRUDService.save(angular.copy(item)));

				requestPromise.then(function(responseItem) {
					item.id = responseItem.id;

					if (responseItem.resourceType === "OperationOutcome") {
						var responseObj = {resource: item};
						if (!that.idGetter(responseObj)) {
							that.dataPointers.list = null;
							return requestPromise;
						}
						that._deleteFromLocalList(responseObj);
						var scopedList = angular.copy(that.dataPointers.list) || [],
							unfilteredList = angular.copy(that.dataPointers.unfilteredList) || [];
						scopedList.push(responseObj);
						unfilteredList.push(responseObj);
						scopedList = that.config.collection.filter ? that.config.collection.filter(scopedList, that.dataPointers.filters) : scopedList;
						scopedList = scopedList.sort(that.config.collection.sortComparator);
						unfilteredList = unfilteredList.sort(that.config.collection.sortComparator);

						that.dataPointers.list = scopedList;
						that.dataPointers.unfilteredList = unfilteredList;
					}
					else {
						fhirDataItem = responseItem;
						fhirDataItem.meta.tag.push({code: "create", system: "http://hl7.org/fhir/action-type"});

						var foundDataItem = false;
						that.dataPointers.list.forEach(function(elem) {
							if (elem.resource.id === fhirDataItem.id) {
								return foundDataItem = true;
							}
						});
						if (!foundDataItem) that.dataPointers.list.push({resource: angular.copy(fhirDataItem)});
						fhirDataItem = null;

						that.dataPointers.list = that.dataPointers.list.sort(that.config.collection.sortComparator);
						for (var tItem in that.dataPointers.list) {
							var tResource = that.dataPointers.list[tItem].resource;
							that.config.object.responseTransform(tResource);
						}
					}
				});
				return requestPromise;
			};
		};

		AbstractGoalsService.prototype = Object.create(CollectionCRUDService.prototype);
		AbstractGoalsService.prototype.constructor = AbstractGoalsService;

		return AbstractGoalsService;
	});
});
